import { NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"
import { rowToCustomer } from "@/lib/utils/db-helpers"

export async function GET(
  request: NextRequest,
  { params }: { params: Promise<{ id: string }> }
) {
  try {
    const { id } = await params

    const customerResult = await query(
      `SELECT * FROM customers WHERE id = $1`,
      [id]
    )

    if (customerResult.rows.length === 0) {
      return NextResponse.json({ error: "Customer not found" }, { status: 404 })
    }

    const customer = rowToCustomer(customerResult.rows[0])

    // Get time entries with user info
    const timeEntriesResult = await query(
      `SELECT te.*, 
        json_build_object(
          'name', u.name,
          'email', u.email
        ) as user
       FROM time_entries te
       LEFT JOIN users u ON u.id = te."userId"
       WHERE te."customerId" = $1
       ORDER BY te."createdAt" DESC`,
      [id]
    )

    const timeEntries = timeEntriesResult.rows.map((row: {
      id: string
      description: string
      startTime: Date
      endTime: Date
      duration: number
      createdAt: Date
      updatedAt: Date
      customerId: string
      userId: string
      user: unknown
    }) => ({
      id: row.id,
      description: row.description,
      startTime: row.startTime,
      endTime: row.endTime,
      duration: row.duration,
      createdAt: row.createdAt,
      updatedAt: row.updatedAt,
      customerId: row.customerId,
      userId: row.userId,
      user: typeof row.user === 'string' ? JSON.parse(row.user) : row.user,
    }))

    return NextResponse.json({
      ...customer,
      timeEntries,
    })
  } catch (error) {
    console.error("Error fetching customer:", error)
    return NextResponse.json({ error: "Failed to fetch customer" }, { status: 500 })
  }
}

export async function PUT(
  request: NextRequest,
  { params }: { params: Promise<{ id: string }> }
) {
  try {
    const { id } = await params
    const body = await request.json()
    const { name, email, phone, company, address, notes } = body

    if (!name) {
      return NextResponse.json({ error: "Name is required" }, { status: 400 })
    }

    const result = await query(
      `UPDATE customers 
       SET name = $1, email = $2, phone = $3, company = $4, address = $5, notes = $6, "updatedAt" = $7
       WHERE id = $8
       RETURNING *`,
      [
        name,
        email || null,
        phone || null,
        company || null,
        address || null,
        notes || null,
        new Date(),
        id,
      ]
    )

    if (result.rows.length === 0) {
      return NextResponse.json({ error: "Customer not found" }, { status: 404 })
    }

    const customer = rowToCustomer(result.rows[0])
    return NextResponse.json(customer)
  } catch (error) {
    console.error("Error updating customer:", error)
    return NextResponse.json({ error: "Failed to update customer" }, { status: 500 })
  }
}

export async function DELETE(
  request: NextRequest,
  { params }: { params: Promise<{ id: string }> }
) {
  try {
    const { id } = await params

    await query(
      `DELETE FROM customers WHERE id = $1`,
      [id]
    )

    return NextResponse.json({ success: true })
  } catch (error) {
    console.error("Error deleting customer:", error)
    return NextResponse.json({ error: "Failed to delete customer" }, { status: 500 })
  }
}
